# Google Sheets Integration Setup Guide

## Overview
This system can automatically sync attendance data to Google Sheets. Follow these steps to set it up.

## Method 1: Using Service Account (Recommended)

### Step 1: Create Google Cloud Project

1. Go to [Google Cloud Console](https://console.cloud.google.com/)
2. Click "Select a project" → "New Project"
3. Enter project name (e.g., "Attendance System")
4. Click "Create"

### Step 2: Enable Google Sheets API

1. In your project, go to "APIs & Services" → "Library"
2. Search for "Google Sheets API"
3. Click on it and press "Enable"

### Step 3: Create Service Account

1. Go to "APIs & Services" → "Credentials"
2. Click "Create Credentials" → "Service Account"
3. Fill in:
   - Service account name: `attendance-sync`
   - Click "Create and Continue"
   - Skip role assignment (click "Continue")
   - Click "Done"

### Step 4: Create Key

1. Click on the service account you just created
2. Go to "Keys" tab
3. Click "Add Key" → "Create new key"
4. Select "JSON" format
5. Click "Create" - JSON file will download

### Step 5: Upload JSON File

1. Rename the downloaded JSON file to `google_service_account.json`
2. Upload it to: `/config/google_service_account.json`
3. **IMPORTANT**: Set file permissions to 644 (readable by web server, not publicly accessible)

### Step 6: Create Google Sheet

1. Create a new Google Sheet
2. Name it "Employee Attendance" (or any name)
3. Add headers in row 1:
   - Column A: Date
   - Column B: Employee Name
   - Column C: Department
   - Column D: Check In Time
   - Column E: Check Out Time

### Step 7: Share Sheet with Service Account

1. In your Google Sheet, click "Share" button
2. Get the service account email from the JSON file (look for `client_email`)
3. Paste the email in the share dialog
4. Give it "Editor" permission
5. Click "Send" (you can uncheck "Notify people")

### Step 8: Get Sheet ID

1. Open your Google Sheet
2. Look at the URL: `https://docs.google.com/spreadsheets/d/SHEET_ID_HERE/edit`
3. Copy the `SHEET_ID_HERE` part

### Step 9: Configure System

Edit `config/google_sheets.php`:

```php
define('GOOGLE_SHEETS_ENABLED', true);
define('GOOGLE_SHEETS_ID', 'YOUR_SHEET_ID_HERE');
define('GOOGLE_SHEETS_RANGE', 'Sheet1!A:E');
```

Replace `YOUR_SHEET_ID_HERE` with the Sheet ID from Step 8.

## Method 2: Using Webhook/API (Alternative)

If you prefer not to use service accounts, you can:

1. Use Google Apps Script to create a webhook endpoint
2. Update `config/google_sheets.php` to send data via HTTP POST
3. Configure the webhook URL

## Testing

After setup:

1. Create a test employee
2. Scan QR code and check in
3. Check your Google Sheet - new row should appear
4. Check out - the row should update with check-out time

## Troubleshooting

### "Service account file not found"
- Verify file is at `/config/google_service_account.json`
- Check file permissions (644)

### "Failed to get access token"
- Verify JSON file is valid
- Check service account email is correct
- Ensure Google Sheets API is enabled

### "Permission denied"
- Make sure you shared the sheet with service account email
- Verify service account has "Editor" permission

### Data not appearing
- Check PHP error logs
- Verify Sheet ID is correct
- Ensure range matches your sheet structure

## Security Notes

- **NEVER** commit `google_service_account.json` to version control
- Keep file permissions restricted (644)
- Consider using environment variables for sensitive data
- Regularly rotate service account keys

## Manual Sync (If Needed)

If automatic sync fails, you can manually export data:

```sql
SELECT 
    a.date,
    e.name,
    e.department,
    a.check_in_time,
    a.check_out_time
FROM attendance a
JOIN employees e ON a.employee_id = e.id
ORDER BY a.date DESC, e.name;
```

Export this to CSV and import to Google Sheets.
